/*******************************************************************************
 * Copyright (c) 2018 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Red Hat Inc. - copied and renamed in jdt.core.manipulation
 *******************************************************************************/
package org.eclipse.jdt.internal.corext.fix;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;

import org.eclipse.ltk.core.refactoring.RefactoringStatus;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;


/**
 * A clean up solves problems in a compilation unit.
 * <p>
 * The clean up is asked for its requirements through a call to {@link #getRequirements()}. The
 * clean up can request an AST and define how to build this AST. It can base its requirements on the
 * options passed through {@link #setOptions(CleanUpOptions)}.
 * </p>
 * <p>
 * A context containing the information requested by the requirements are passed to
 * {@link #createFix(CleanUpContext)}. A fix capable of fixing the problems is returned by this
 * function if {@link #checkPreConditions(IJavaProject, ICompilationUnit[], IProgressMonitor)} has
 * returned a non fatal error status.
 * </p>
 * <p>
 * At the end {@link #checkPostConditions(IProgressMonitor)} is called.
 * </p>
 * 
 * @since 3.5
 */
public interface ICleanUpCore {

	/**
	 * Sets the options that will be used.
	 * 
	 * @param options the options to use
	 */
	void setOptions(CleanUpOptions options);

	/**
	 * Human readable description for each step this clean up will execute.
	 * <p>
	 * <strong>Note:</strong> This method must only be called after the options have been set.
	 * </p>
	 * 
	 * @return descriptions an array of {@linkplain String strings} or <code>null</code>
	 */
	String[] getStepDescriptions();

	/**
	 * The requirements of this clean up.
	 * <p>
	 * <strong>Note:</strong> This method must only be called after the options have been set.
	 * </p>
	 * 
	 * @return the requirements used for {@link #createFix(CleanUpContext)} to work
	 */
	CleanUpRequirements getRequirements();

	/**
	 * After call to checkPreConditions clients will start creating fixes for
	 * <code>compilationUnits</code> in <code>project</code> unless the result of checkPreConditions
	 * contains a fatal error
	 * 
	 * @param project the project to clean up
	 * @param compilationUnits an array of compilation units to clean up, all member of <code>project</code>
	 * @param monitor the monitor to show progress
	 * @return the result of the precondition check
	 * @throws CoreException if an unexpected error occurred
	 */
	RefactoringStatus checkPreConditions(IJavaProject project, ICompilationUnit[] compilationUnits, IProgressMonitor monitor) throws CoreException;

	/**
	 * Create an <code>ICleanUpFixCore</code> which fixes all problems in <code>context</code> or
	 * <code>null</code> if nothing to fix.
	 * 
	 * @param context a context containing all information requested by {@link #getRequirements()}
	 * @return the fix for the problems or <code>null</code> if nothing to fix
	 * @throws CoreException if an unexpected error occurred
	 */
	ICleanUpFixCore createFix(CleanUpContext context) throws CoreException;

	/**
	 * Called when done cleaning up.
	 * 
	 * @param monitor the monitor to show progress
	 * @return the result of the postcondition check, not null
	 * @throws CoreException if an unexpected error occurred
	 */
	RefactoringStatus checkPostConditions(IProgressMonitor monitor) throws CoreException;

}
